#include <afx.h>
#include <afxtempl.h>
#include "parse.h"
#include "topiclog.h"
#include "fmtspec.h"
#include "docexpr.h"
#include "cmdargs.h"
#include "extract.h"
#include "parsesrc.h"
#include "parsetag.h"
#include "parsejava.h"
#include "parsevb.h"
#include "errmsg.h"

const char *ParseTagField(const char *szCur, char *szTempBuf, CTag *ptagNew, int i, int nTagFields);

//@doc


/*
@func Allocates a new CTag object and adds it to the tag list.

@parm List in which to store the tag.
@parm Line number of tag.
@parm Tag name.

@rdesc The new tag.
*/
CTag *AddTag(CTagList &listTags, long lLineTag, const char *szTag)
{
    // Allocate a TAG structure

	CTag *ptagNew = new CTag;
	
	listTags.AddTail(ptagNew);

    // Record filename, starting line number, tag name

    ptagNew->m_lSrcLineNum  = lLineTag;
    ptagNew->m_sTag = szTag;

	return ptagNew;
}


/*
@func CTag *| ParseTag | This function converts a text block into a <t TAG>
structure containing a list of fields for the tag. The <t TAG> structure is
added to the linked list of <t TAG> structures passed as a parameter.

@rdesc Returns 0 on success or an error code.
*/

int ParseTag(
	CSrcInput &in,          //@parm Input file
    FormatInfo &fmt,        //@parm Formatting structures
	BOOL bAux,				//@parm Whether this is aux tag
	CTag *ptagAuxParent,	//@parm Paragraph tag, parent of aux tag
	BOOL bTopic)            //@parm Whether this is a topic tag
{
    int nTagFields;
    int nExtrFields = 0;
    int nSrcLineNum = in.m_lCurLine;
    int nParseType;

	CPendingList &listPend = bAux ? in.m_pendAux : in.m_pendMain;
	CTagList &listTags = bAux ? in.m_listAuxTags : in.m_listTags;
	const char *szTag  = in.m_sTag;

    const char *szCur;

    CFmtTopic  *pfmtTopic = NULL;
    CFmtPara  *pfmtPara = NULL;

    char *szTempBuf;
    
    if(bTopic)
    {
        CFmtSrchTopic srch;

        srch.m_sName = szTag;
        pfmtTopic = (CFmtTopic *)fmt.topic.Get(&srch);
        if(pfmtTopic == NULL)
            return warnUnknownTopicTag;
    }
    else
    {
        CFmtSrchPara srch;

        srch.m_sName = szTag;
        pfmtPara = (CFmtPara *)fmt.paragraph.Get(&srch);
        if(pfmtPara == NULL)
            return warnUnknownParaTag;
    }

    // Allocate a temporary buffer for storing field text

    szTempBuf = new char[in.m_szTagCur - in.m_achTagBuf + 1];

	CTag *ptagNew = AddTag(listTags, in.m_nLineTag, szTag);

    // Set up a couple fields for the tag, these are topic/para
    // specific. The rest is generic.

    if(bTopic)
    {
        ASSERT(pfmtTopic);
        nTagFields = pfmtTopic->GetNumFields();
        nParseType = pfmtTopic->GetParseType();
    }
    else
    {
        ASSERT(pfmtPara);
        ptagNew->m_nState.IsExample = pfmtPara->IsExampleTag();
        nTagFields = pfmtPara->GetNumFields();
        nParseType = pfmtPara->GetParseType();
    }

    szCur = in.m_achTagBuf;

    // Get the text for each field (separated by chTagFieldSep character)

    for(ptagNew->m_nFields = 0; *szCur && ptagNew->m_nFields < nTagFields; 
    	ptagNew->m_nFields++)
    {
        // Process whitespace between fields, unless this is the
        // final field of an example tag.

        if(!(ptagNew->m_nState.IsExample && (ptagNew->m_nFields+1 == nTagFields)))
        {
            szCur = EatWhiteNl(szCur, &nSrcLineNum);
        }

        // Grab the field

        szCur = ParseTagField(szCur, szTempBuf, ptagNew, ptagNew->m_nFields, nTagFields);

        if(*szCur == chTagFieldSep && ptagNew->m_nFields != nTagFields-1)
            szCur++;
    }

    // Determine whether we have the targeted number of fields. If not, we can
	// try to parse text from the source line preceding the tag.
    
    int nRet = 0;

    if(nTagFields == ptagNew->m_nFields)
    {
        // Check for field overflow

        if(*szCur == chTagFieldSep)
            nRet = warnTooManyFields;
    }
    else
    {
        ASSERT(ptagNew->m_nFields < nTagFields);

		// If fields are missing, need to extract the missing fields from the
		// source. If the tag supports source parsing, first see if the source
		// text preceding the comment block (stored in in.m_achCommentStart
		// on a previous loop) has what is needed. If there is no source text
		// stored there, add the tag to the parsing cue, to parse source text
		// from a later line.
			
		if(nParseType == 0)
		{
            nRet = warnWrongNumFields;
		}
		else
		{
			in.CueParse(listPend, ptagNew, nTagFields, nParseType, bTopic);
		}

    }
    
    delete szTempBuf;

    return nRet;
}




/*
@func char *| ParseTagField | This function creates a memory buffer
containing the text of a tag field. 

@rdesc Returns a pointer to the first character following the field.
This is either the end of the buffer or the pipe character following
the field.
*/

const char *ParseTagField(
    const char *szTagBuf, 	//@parm Pointer to the buffer containing the tag text.
    char *szTempBuf, 	    //@parm Pointer to a temporary buffer for use during parsing.
    CTag *ptagNew, 		    //@parm Pointer to the <t TAG> structure to add the field to.
    int i, 				    //@parm Index of field we're adding.
    int nTagFields)		    //@parm Count of tag fields as defined in format file.
{
    int nFieldLen;

    // If an example field, just copy the whole string into the
    // field.

    if( ptagNew->m_nState.IsExample && (i+1 == nTagFields) )
    {
        const char *szCont = szTagBuf + strlen(szTagBuf);

        const char *szT = TrimWhite(szCont, szTagBuf);

        nFieldLen = szT-szTagBuf+1;

        ptagNew->m_aszFieldText[i] = new char[nFieldLen + 1];
        memcpy(ptagNew->m_aszFieldText[i], szTagBuf, nFieldLen);
        ptagNew->m_aszFieldText[i][nFieldLen] = '\0';
		ptagNew->m_anFieldLen[i] = nFieldLen;

        return szCont;
    }

    // Otherwise, copy character by character, dealing with escaped pipes and 
    // newlines...

    for(char *szCur = szTempBuf; *szTagBuf && *szTagBuf != '|'; szCur++, szTagBuf++)
    {
        switch(*szTagBuf)
        {
        case '\\':

            // Ignore an escaped pipe character, just copy as literal

            if(*(szTagBuf+1) == '|')
                szTagBuf++;

            break;

        case '\r':
        case '\n':

            // Delete trailing whitespace at the end of a line

            while((szCur > szTempBuf) && 
            	  (*(szCur-1) == chSpace || *(szCur-1) == chTab) )
			{
                szCur--;
			}

            break;
        }

        *szCur = *szTagBuf;
    }

    // Delete trailing whitespace and newlines from
    // temp buffer containing field

    while((szCur > szTempBuf) && (isspace(*(szCur-1))))
        szCur--;

    *szCur = '\0';

    nFieldLen = strlen(szTempBuf);
    ptagNew->m_aszFieldText[i] = new char[nFieldLen+1];
    memcpy(ptagNew->m_aszFieldText[i], szTempBuf, nFieldLen);
    ptagNew->m_aszFieldText[i][nFieldLen] = '\0';
	ptagNew->m_anFieldLen[i] = nFieldLen;
    
    return szTagBuf;
}


const char *SkipTemplate(
	const char *szT)
{
	if(strncmp(szT, "template", 8) == 0)
	{
		szT += 8;
		szT = SeekChar(szT, chOpenAngle);
		if(szT == NULL)
			return szT;

		szT = MatchParen(szT, chCloseAngle);
		if(*szT != chCloseAngle)
			return NULL;

		szT = EatWhite(++szT);
	}

	return szT;
}


const char *ParseOperator(
	const char *szName,
    CString &sOp)
{
	const char *szT;
	const char *szAngle;

	ASSERT(szName);

    if(strncmp(szName, "operator", 8) == 0)
    {
        sOp = "operator";
        szT = EatWhite(szName+8);

		// Look at various operator types

		if(*szT == chOpenParen)
		{
			// operator() (we may have already null-terminated the first 
			// open paren)

            sOp += "()";
			szT = EatWhite(++szT);
			if(*szT == chCloseParen)
				szT++;
		}
		else if(strncmp(szT, "delete", 6) == 0)
		{
			// operator delete

            sOp += " delete";

			szT += 6;
		}
		else if(strncmp(szT, "new", 3) == 0)
		{
			// operator new

			sOp += " new";
			szT += 3;
		}
		else if(isalpha(*szT))
		{
			// e.g. operator const BITMAPINFOHEADER *

			szName = szT;
			while(*szT && *szT != chOpenParen && *szT != chOpenAngle)
				szT++;

			sOp += chSpace;
			sOp += CString(szName, szT-szName);
		}
		else
		{
			// other operators e.g. operator+

            szName = szT;

            while(*szT && *szT != ' ' && *szT != chOpenParen)
			{
				// An open angle bracket might indicate a 
				// template argument - see if there is a 
				// matching close angle bracket.

				if(*szT == chOpenAngle)
				{
					szAngle = MatchParen(szT, chCloseAngle, 0);
					if(*szAngle == chCloseAngle)
						break;
				}
                szT++;
			}

			sOp += CString(szName, szT-szName);
        }

        szT = EatWhite(szT);
        if(*szT == chOpenAngle)
        {
            const char *szTempl = szT;
            szT = MatchParen(szT, chCloseAngle);
            if(*szT == chCloseAngle)
            {
                szT++;
                sOp += CString(szTempl, szT-szTempl);
            }
        }

		return EatWhite(szT);
    }
	else
	{
		return NULL;
	}
}


//@func Skips past a set of template args enclosed
// in angle brackets.
//
//@rdesc Pointer to closing template bracket, or
// one character past starting point if no close
// bracket was found.

const char *SkipTemplateArgs(
	const char *szSrc)			//@parm Pos of opening bracket
{
	const char *szT = MatchParen(szSrc, chCloseAngle);
	if(*szT == chCloseAngle)
	{
		return szT+1;
	}
	else
	{
		return szSrc+1;
	}
}


//@func Parse a C++ constant or #define.
//@parm Tag to populate.
//@parm Pointer to source string with constant declaration.
//
//@comm Look for: const <type> <name> = <value>
//
// Or: #define <name>

int ParseConstant(
	CTag *ptag,
	const char *szSrc)
{
    static char szStopConst[] = "=;\r\n/";
	const char *szName = NULL;
	const char *szType = NULL;

	// Skip const keyword

	szSrc = EatWhite(szSrc);

	// Deal with #defines on separate track (thank you
	// Casey (klim@sgi.net).

    if ( strncmp( szSrc, "#define", 7 ) == 0 )
    {
        szType = szSrc;                 // type is "#define"
        szSrc = EatWhite(szSrc+7);      // skip to name

        szName = szSrc;                 // start of name
        while( isidchar(*szSrc) )       // search for end of name
            szSrc++;

        if ( szName == szSrc )          // check for null name
            return warnSrcFieldsIncomplete;

        goto AddFields;
    }

	if(strncmp(szSrc, "const", 5) != 0)
		return warnSrcFieldsUnavailable;

	szSrc = szSrc+5;

    // Get type and name
    
    szSrc = SkipTemplate(EatWhite(szSrc));

    // Look for tokens until we hit a separator, paren, newline,
    // or comment

    while(!(*szSrc == '\0' || strchr(szStopConst, *szSrc)))
    {
        if(isidchar(*szSrc))
        {
        	if(szType == NULL)
                szType = szSrc;

            szName = szSrc;

            while(isidchar(*szSrc))
	            szSrc++;
	    }
	    else if(*szSrc == chOpenBracket)
        {
            szSrc = MatchParen(szSrc, chCloseBracket);
            if(*szSrc)
                szSrc++;
        }
		else
        {
        	szName = NULL;
        	szSrc++;
        }

		// Get the next token.
	
        szSrc = EatWhite(szSrc);
    }

AddFields:

    // If we didn't find either of the tokens, parsing failed, exit.

    if(NULL == szName || NULL == szType || szName == szType)
		return warnSrcFieldsIncomplete;

    // Move existing fields over.
    
    ptag->ShiftFields(0, 2);

    // Copy constant name

	ptag->CopySrcString(1, szName, szSrc);

    // Copy type.

    ptag->CopySrcString(0, szType, szName);

	return 0;
}



//@func Parses a class declaration of the form
//
// class __far MyClass:public YourClass

int ParseClass(
	CTag *ptag,     			//@parm Where to copy name
	const char *szSrc)		    //@parm Start of class declaration
{
	const char *szName = NULL;  				//Class name
	const char *szClose;						//End of class name
	const char *szClassStop = ",{/:";			//Stop chars for class

	// Skip the template, if present

	szSrc = SkipTemplate(EatWhite(szSrc));
	if(szSrc == NULL)
		return warnSrcFieldsIncomplete;

  	// Find "class"

	szSrc = EatWhite(szSrc);
	
	if(_strnicmp(szSrc, "class ", 6) && _strnicmp(szSrc, "class\t", 6))
		return warnSrcFieldsUnavailable;
		
	// Advance past "class" and find class name.
	
	szSrc = EatWhite(szSrc + 6);
	
	while(*szSrc && !strchr(szClassStop, *szSrc))
	{
		szName = szSrc;
		
    	if(!isidchar(*szName))
			return warnSrcFieldsIncomplete;
    	
    	while(isidchar(*szSrc))
    		szSrc++;

		szClose = szSrc;

        szSrc = EatWhite(szSrc);
	}
	
	if(szName == NULL)
		return warnSrcFieldsIncomplete;
    	
	ptag->ShiftFields(0, 1);

	ptag->CopySrcString(0, szName, szClose);
	
	return 0;	
}



int ParseParameter(
	CTag *ptag,
	CParseText &txt,
	int nParseType,
	BOOL bReqTerm)
{
    static char szStopParm[] = ",;)\r\n/'";
	static char szEndParm[] = ",;()";
	static char szSkipChars[] = ",({";
	const char *szName = NULL;
//bugfix
	const char *szNameEnd = NULL;
	const char *szType = NULL;
	const char *szT;
    const char *szValue = NULL;
    const char *szValueEnd = NULL;

    if(txt.m_szCur == NULL)
		txt.m_szCur = txt.m_szBase;

	txt.m_szCur = EatWhite(txt.m_szCur);
	if(strchr(szSkipChars, *txt.m_szCur))
		txt.m_szCur = EatWhite(txt.m_szCur+1);

	txt.m_szCur = SkipTemplate(EatWhite(txt.m_szCur));

    // Look for tokens until we hit a separator, paren, newline,
    // or comment

    while(!(*txt.m_szCur == '\0' || strchr(szStopParm, *txt.m_szCur)))
    {
        if(isidchar(*txt.m_szCur))
        {
        	if(szType == NULL)
                szType = txt.m_szCur;

            szName = txt.m_szCur;
			szNameEnd = NULL;

            while(isidchar(*txt.m_szCur))
	            txt.m_szCur++;
	    }
	    else if(*txt.m_szCur == chOpenBracket)
        {
            txt.m_szCur = MatchParen(txt.m_szCur, chCloseBracket);
            if(*txt.m_szCur)
                txt.m_szCur++;
        }
        else if(nParseType == CFmtPara::parseParamOpt && chEqual == *txt.m_szCur)
        {
            // mark end of name
			szNameEnd = txt.m_szCur;

			// advance past equals
            txt.m_szCur = EatWhite(++txt.m_szCur);

            // get the value name
            if(!isidchar(*txt.m_szCur))
                return warnSrcFieldsMultiline;

            szValue = txt.m_szCur;
            while(isidchar(*txt.m_szCur))
                txt.m_szCur++;

            szValueEnd = txt.m_szCur;
        }
	    else if(*txt.m_szCur == chOpenParen)
        {
			// Determine whether this is the opening for the parameter
			// list. If the parent doesn't have a matching close paren
			// on the same line, break out.

            szT = MatchParen(txt.m_szCur, chCloseParen);
			if(*szT != chCloseParen)
				break;

			// Search past the open paren. If there is no token following
			// the paren, break out.
			//
			// @devnote If there is some weird character following the paren,
			// like *&, this will break - but that would be a very strange
			// construct.
			
			szT = EatWhite(++szT);
			if(!(isidchar(*szT) || *szT == chOpenParen))
				break;

			txt.m_szCur = szT;
        }
		else if(*txt.m_szCur == chOpenAngle)
		{
			szT = SkipTemplateArgs(txt.m_szCur);

			szNameEnd = txt.m_szCur;

			txt.m_szCur = szT;
		}
		else
        {
        	szName = NULL;
        	txt.m_szCur++;
        }

		// Get the next token.
	
        txt.m_szCur = EatWhite(txt.m_szCur);
    }

	// See if we found all of it.

	if(bReqTerm && (*txt.m_szCur == '\0' || !strchr(szEndParm, *txt.m_szCur)))
		return warnSrcFieldsMultiline;
    
	// If we didn't find any tokens, parsing failed, exit.

    if(NULL == szName && NULL == szType)
		return warnSrcFieldsIncomplete;

	if(szName == szType)
		szType = NULL;

    // Move existing fields over.
    
    if(CFmtPara::parseParamOpt == nParseType)
    {
        ptag->ShiftFields(0, 3);
    }
    else
    {
        ptag->ShiftFields(0, 2);
    }

    // COPY FUNCTION NAME

	ptag->CopySrcString(1, szName, szNameEnd ? szNameEnd : txt.m_szCur);

    // Copy the type.

    if(szType)
    	ptag->CopySrcString(0, szType, szName);
	else
		ptag->CopySrcString(0);

    // Copy value.

    if(CFmtPara::parseParamOpt == nParseType)
    {
        if(szValue)
            ptag->CopySrcString(2, szValue, szValueEnd);
        else
            ptag->CopySrcString(2);
    }


/*bug
	if(nParseType == CFmtTopic::parseFunc)
		txt.m_szBase = txt.m_szCur;
bug*/
	return 0;	
}


void AdjustParmWhiteSpace(
	char *szParm)
{
	int i, j;

	CString sParmCopy(szParm);
    char *szParmCopy = sParmCopy.GetBuffer(sParmCopy.GetLength());

	for(i = 0, j = 0; szParmCopy[i]; j++)
	{
		if(iswhite(szParmCopy[i]))
		{
			szParm[j] = chSpace;
			while(iswhite(szParmCopy[i]))
				i++;
		}
		else
		{
			szParm[j] = szParmCopy[i++];
		}
	}
	szParm[j] = '\0';

	sParmCopy.ReleaseBuffer();
}


int ParseMember(
	CTag *ptag,
	CParseText &txt,
	BOOL bCopyParams)
{
    static char szStopMember[] = "\r\n/=;{";
	static char szEndMember[] = ";{=:";
	const char *szT;
	const char *szClose;
	const char *szName = NULL;
	const char *szType = NULL;
    const char *szParams = NULL;
    CString sOp;

    const char *szSrc = EatWhite(txt.m_szBase);
	if(chOpenCurly == *szSrc)
		szSrc = EatWhite(szSrc+1);

    // Look for tokens until we hit a separator, paren, newline,
    // or comment

    while(!(*szSrc == '\0' || strchr(szStopMember, *szSrc)))
    {
        // Name token

        if(isidchar(*szSrc) || *szSrc == chTilde)
        {
        	if(szType == NULL)
                szType = szSrc;

            szName = szSrc;

            // See if we have an operator... function - these
			// need special treatment.

            if(szT = ParseOperator(szSrc, sOp))
			{
				szSrc = szT;
			}
			else
			{
				if(*szSrc == chTilde)
					szSrc++;
				while(isidchar(*szSrc))
	            	szSrc++;
			}
	    }
        // Parameter list or macro return value

		else if(*szSrc == chOpenParen)
		{
			// Find matching paren and determine whether this is the parameter
			// list we're looking at.

			szClose = MatchParen(szSrc, chCloseParen);
			if(*szClose != chCloseParen)
				return warnSrcFieldsMultiline;

            szParams = szSrc;

			// If a paren is found, and the next character after the paren is a
			// stop character, the parens were the last on the line - parameter
			// list.

			szT = EatWhite(++szClose);
			if(*szT == '\0' || strchr(szStopMember, *szT))
			{
                szSrc = szT;
			}
			else if(strncmp(szT, "const", 5) == 0)
			{
				szSrc = szClose = szT+5;
			}
			else
			{
				// Continue past closing paren
				szSrc = szClose+1;
			}
		}
	    
	    // Array specifier

	    else if(*szSrc == chOpenBracket)
        {
            szT = MatchParen(szSrc, chCloseBracket);
            if(*szT == chCloseBracket)
                szSrc = szT+1;
			else
				return warnSrcFieldsMultiline;
        }

		// Template specifier

		else if(*szSrc == chOpenAngle)
		{
			szSrc = SkipTemplateArgs(szSrc);
		}
        
        // Non-name token: must keep looking for the real name
        
        else
        {
        	if(szType && strncmp(szType, "operator", 8) != 0)
				szName = NULL;
        	szSrc++;
        }

		// Get the next token.
	
        szSrc = EatWhite(szSrc);
    }

    if(!strchr(szEndMember, *szSrc))
        return warnSrcFieldsMultiline;

    // If we didn't find any tokens, parsing failed, exit.

    if(NULL == szName && NULL == szType)
		return warnSrcFieldsIncomplete;

	if(szName == szType)
		szType = NULL;

    // Move existing fields over.
    
    if(bCopyParams)
    {
        ptag->ShiftFields(0, 3);
    }
    else
    {
        ptag->ShiftFields(0, 2);
    }

    // Copy function name. Variations:
	// - with operator
	//		- with or without template args
	// - without operator

	if(szName)
	{
		// See if operator function.
		if(sOp.IsEmpty())
		{
			ptag->CopySrcString(1, szName, szParams ? szParams : szSrc);
		}
		else
		{
			ptag->CopySrcString(1, sOp);
		}
	}

	// Copy type

	if(szType)
		ptag->CopySrcString(0, szType, szName);
	else
		ptag->CopySrcString(0);

    // IF PARAMETER LIST, COPY IT AS WELL

    if(bCopyParams && szParams)
	{
        ptag->CopySrcString(2, szParams, szClose);
		AdjustParmWhiteSpace(ptag->m_aszFieldText[2]);
	}

    // Set parse resume at parameter list, in case there are
    // parameter tags too.

	if(szParams)
	{
		txt.m_szCur = szParams;
	}
	
	return 0;
}



int ParseMfunc(
	CTag *ptag,
	CParseText &txt)
{
	CString sOp;

    const char *szT = NULL;

    const char *szClass = NULL;
    const char *szClassEnd = NULL;
	const char *szType;
	const char *szName;
    const char *szNameEnd = NULL;

    //** SET: szType  TO: first char on line

	szType = SkipTemplate(EatWhite(txt.m_szCur));
	if(szType)
	{
        szType = EatWhite(szType);
    }
	else
    {
    	return warnSrcFieldsIncomplete;
    }

    // Get the member function separator. Note that there might be
    // a :: in the return type (for class enums).

    szName = strstr(szType, "::");
    if(szName == NULL)
		return warnSrcFieldsMultiline;

    szT = strstr(szName+2, "::");
    if(szT)
        szName = szT;

    // Locate the class name.

    szClass = szName;
	
	// May have to skip over a template specifier.

	if(*(szClass-1) == chCloseAngle)
	{
		while(szClass != szType && *szClass != chOpenAngle)
			szClass--;
		if(*szClass != chOpenAngle)
			return warnSrcFieldsMultiline;
		
		// Null-terminate at end of template specifier.
		szClassEnd = szClass;
	}

    // Search backward for beginning of class name.

    while(isidchar(*(szClass-1)) && szClass-1 >= szType)
        szClass--;
    if(szT == szClass)
		return warnSrcFieldsMultiline;

    // Null-terminate class name, advance to function name

    if(szClassEnd == NULL)
        szClassEnd = szName;

    //** SET: szName   TO: first char of function name

    szName += 2;

    // Find end of function name, null-terminate it.

    for(szT = szName; *szT && !strchr("<( \r\n", *szT); szT++);

    // See if this is an overloaded operator - if not, null-terminate
	// the function name.

	const char *sz = ParseOperator(szName, sOp);
	if(sz)
	{
		szNameEnd = sz;
	}
	else
    {
        szNameEnd = szT;
    }

    // Shift tag fields over. First record class name & function
    // name

    ptag->ShiftFields(0, 3);

	ptag->CopySrcString(1, szClass, szClassEnd);

    if(sOp.IsEmpty())
    	ptag->CopySrcString(2, szName, szNameEnd);
    else
    	ptag->CopySrcString(2, sOp);

    // Store type declaration.

	ptag->CopySrcString(0, szType, szClass);

	txt.m_szCur = szNameEnd;

	return 0;
}


//@func Parses the structure tag name from a new-style C++
// enum/structure definition. Places tag name in field 0.

//@rdesc Returns 0 on success or an error code.

int ParseStruct(
	CTag *ptag,			//@parm Tag structure to fill
	const char *szSrc,	//@parm Source string
	int nParseType)		//@parm Either CFmtTopic::parseStruct or parseEnum

{
	const char *szName = NULL;
	static char szRUnion[] = "union";
	static char szREnum[] = "enum";
	static char szRStruct[] = "struct";
	static char szRFriend[] = "friend";
	char *szElemName;

	switch(nParseType)
	{
	case CFmtTopic::parseUnion:
		szElemName = szRUnion;
		break;

	case CFmtTopic::parseStruct:
		szElemName = szRStruct;
		break;

	case CFmtTopic::parseEnum:
		szElemName = szREnum;
		break;

	case CFmtPara::parseFriend:
		szElemName = szRFriend;
		break;

	default:
		ASSERT(0);
		szElemName = szRStruct;
		break;
	}

  	// Find "enum"

	szSrc = EatWhite(szSrc);
	
	if(_strnicmp(szSrc, szElemName, strlen(szElemName)))
		return warnSrcFieldsUnavailable;
		
	// Advance past "enum" and enum name
	
	szSrc = EatWhite(szSrc + strlen(szElemName));

	if(!isidchar(*szSrc))
		return warnSrcFieldsIncomplete;

	szName = szSrc;
	
	while(isidchar(*szSrc))
		szSrc++;

	ptag->ShiftFields(0, 1);

	ptag->CopySrcString(0, szName, szSrc);
	
	return 0;
}


//@func Parses an enumeration member name.
// Name is placed in field 0 of tag.

//@rdesc Zero on success or error code

int ParseEmem(
	CTag *ptag,				//@parm Tag to fill
	const char *szSrc)		//@parm Source string
{
	const char *szName = NULL;
			
    szSrc = EatWhite(szSrc);
	if(chOpenCurly == *szSrc)
		szSrc = EatWhite(szSrc+1);

	szName = szSrc = EatWhite(szSrc);
	if(!isidchar(*szSrc))
		return warnSrcFieldsIncomplete;
	
	while(isidchar(*szSrc))
		szSrc++;
	
	ptag->ShiftFields(0, 1);

	ptag->CopySrcString(0, szName, szSrc);

	return 0;
}


/**************************************************
 ParseSrc

@func This function retrieves autoduck tags from C constructs within
the source file.

@rdesc Returns zero if successful or 1 if unsuccessful.
*/
int ParseSrc(
	CTag *ptag, 	    // @parm Specifies a pointer to the <t TAG> structure to add
					    // the field to.
	CParseText &txt, 	// @parm Specifies a pointer to the C source information.
	int nSrcLine, 	    // @parm Specifies the line number of the C source information.
	int nParseType,     // @parm Type of source parsing requested.
	int bReqTerm)       // @parm Whether to require a terminating character
{
    int nRet;

	if(txt.m_szCur == NULL)
		txt.m_szCur = txt.m_szBase;

    switch(nParseType)
    {

	// *******************************************
	// C/C++ Topic stuff

    case CFmtTopic::parseFunc:

    	nRet = ParseParameter(ptag, txt, nParseType, bReqTerm);

        break;

    case CFmtTopic::parseConstant:

    	nRet = ParseConstant(ptag, txt.m_szCur);

    	break;

    case CFmtTopic::parseMfunc:

    	nRet = ParseMfunc(ptag, txt);

    	break;

    case CFmtTopic::parseClass:
	
    	nRet = ParseClass(ptag, txt.m_szCur);

    	break;
		
    case CFmtTopic::parseUnion:
    case CFmtTopic::parseEnum:
	case CFmtTopic::parseStruct:
	case CFmtPara::parseFriend:

    	nRet = ParseStruct(ptag, txt.m_szCur, nParseType);

    	break;

	// C/C++ paragraph stuff

	case CFmtPara::parseParam:
	case CFmtPara::parseParamOpt:
    case CFmtPara::parseField:

    	nRet = ParseParameter(ptag, txt, nParseType, bReqTerm);

        break;

    case CFmtPara::parseMember:

    	nRet = ParseMember(ptag, txt, TRUE);

    	break;

    case CFmtPara::parseMeth:

    	nRet = ParseMember(ptag, txt, FALSE);

    	break;

    case CFmtPara::parseEmem:

    	nRet = ParseEmem(ptag, txt.m_szCur);

    	break;

	// *******************************************
	// Java topics

	case CFmtTopic::parseJClass:
		 nRet = ParseJavaClass(ptag, txt, TRUE);

		 break;

	case CFmtTopic::parseJInterface:
		 nRet = ParseJavaClass(ptag, txt, FALSE);

		 break;

	// Java paragraphs

	case CFmtPara::parseJMethod:
		 nRet = ParseJavaMethod(ptag, txt);

		 break;

    case CFmtPara::parseJParam:

    	nRet = ParseJavaParameter(ptag, txt);

    	break;

	// *******************************************
	// VB topics
	//
	// these all go through the same routine.
	// parameter 3 TRUE: grab the parameter list
	// parameter 4 TRUE: grab the type

	case CFmtTopic::parseBtype:
		nRet = ParseVbRoutine(ptag, txt, FALSE, FALSE);
		break;

	case CFmtTopic::parseBfunc:
		nRet = ParseVbRoutine(ptag, txt, FALSE, TRUE);
		break;

	case CFmtTopic::parseBsub:
		nRet = ParseVbRoutine(ptag, txt, FALSE, FALSE);
		break;

	// VB paragraphs 

	case CFmtPara::parseBfield:
		nRet = ParseVbField(ptag, txt);
		break;

	case CFmtPara::parseBparam:
		nRet = ParseVbParm(ptag, txt);
		break;

	case CFmtPara::parseBProperty:
		nRet = ParseVbProperty(ptag, txt);
		break;
    
	// shouldn't be here.
	
	default:
		
		ASSERT(0);
		nRet = fmterrBadSrcParse;
		break;


    }

    return nRet;
}



